unit sys_main;

interface

{
        SysUtils
   ?         ...
  :
   -  
   -  Integer  String  
   -   (try except finally)
   -      
}

uses
  Windows;

const
  NULL_FILE = INVALID_HANDLE_VALUE;

type
  TFile = DWORD;
  TByteArray = array [0..1023] of Byte;
  PByteArray = ^TByteArray;

// Time
  function GetTime: Integer;
// Int convert
  function IntToStr(Value: Integer): string;
  function StrToInt(const S: string): Integer;
  function StrToIntDef(const S: string; Default: Integer): Integer;
// strings
  function LowerCase(const s: string): string;
// FileNames
  function ExtractFileExt(const FileName: string): string;

  function FileOpen(FileName: PChar; Rewrite: Boolean = False): TFile;
  function FileValid(F: TFile): Boolean;
  procedure FileClose(var F: TFile);
  procedure FileFlush(F: TFile);
  function FileWrite(F: TFile; const Buf; Count: DWORD): DWORD;
  function FileRead(F: TFile; var Buf; Count: DWORD): DWORD;
  procedure FileSeek(F: TFile; Pos: Integer);
  function FileSize(F: TFile): DWORD;
  function FilePos(F: TFile): DWORD;
  function FileExists(FileName: PChar): Boolean;

function StrLen(const Str: PChar): Cardinal;
function StrLComp(const Str1, Str2: PChar; MaxLen: Cardinal): Integer; assembler;
procedure StrDispose(Str: PChar);
function StrLCopy(Dest: PChar; const Source: PChar; MaxLen: Cardinal): PChar; assembler;
function StrAlloc(Size: Cardinal): PChar;
function StrMove(Dest: PChar; const Source: PChar; Count: Cardinal): PChar;
function StrNew(const Str: PChar): PChar;

type
  TInterface = class(TInterfacedObject)
    constructor CreateEx;
  end;

  TIntStatic = class(TInterface)
  protected
    function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  end;

  TStream = class
    procedure Seek(Pos: Integer); virtual; abstract;
    function Valid: Boolean; virtual; abstract;
    function Read(var Buf; Count: Integer): Integer; virtual; abstract;
    function Size: Integer; virtual; abstract;
    function Pos: Integer; virtual; abstract;
  end;

  TFileStream = class(TStream)
    constructor Create(FileName: string);
    destructor Destroy; override;
   private
    F : TFile;
   public
    procedure Seek(Pos: Integer); override;
    function Valid: Boolean; override;
    function Read(var Buf; Count: Integer): Integer; override;
    function Size: Integer; override;
    function Pos: Integer; override;
  end;

  TMemoryStream = class(TStream)
    constructor Create(Mem: Pointer; Size: Integer);
   private
    FMem  : Pointer;
    FSize : Integer;
    FPos  : Integer;
   public
    procedure Seek(Pos: Integer); override;
    function Valid: Boolean; override;
    function Read(var Buf; Count: Integer): Integer; override;
    function Size: Integer; override;
    function Pos: Integer; override;
  end;


implementation

//==    
function GetTime: Integer;
var
  T : LARGE_INTEGER;
  F : LARGE_INTEGER;
begin
  QueryPerformanceFrequency(Int64(F));
  QueryPerformanceCounter(Int64(T));
  Result := Trunc(1000 * T.QuadPart / F.QuadPart);
end;

//== Int
function IntToStr(Value: Integer): string;
begin
  Str(Value, Result);
end;

function StrToInt(const S: string): Integer;
var
  er : Integer;
begin
  Val(S, Result, er);
end;

function StrToIntDef(const S: string; Default: Integer): Integer;
var
  er : Integer;
begin
  Val(S, Result, er);
  if er = 0 then
    Result := Default;
end;

function LowerCase(const s: string): string;
var
  i, l   : integer;
  Rc, Sc : PChar;
begin
  l := Length(s);
  SetLength(Result, l);
  Rc := Pointer(Result);
  Sc := Pointer(s);
  for i := 1 to l do
  begin
    if s[i] in ['A'..'Z', ''..''] then
      Rc^ := Char(Byte(Sc^) + 32)
    else
      Rc^ := Sc^;
    inc(Rc);
    inc(Sc);
  end;
end;

function ExtractFileExt(const FileName: string): string;
var
  i : Integer;
begin
  for i := Length(FileName) downto 1 do
    if FileName[i] = '.' then
    begin
      Result := Copy(FileName, i + 1, Length(FileName));
      Exit;
    end;
  Result := '';
end;

//===    ===--------------------------------------------
//  
function FileOpen(FileName: PChar; Rewrite: Boolean): TFile;
begin
  if Rewrite then
    Result := CreateFile(FileName, GENERIC_ALL, 0, nil, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0)
  else
    Result := CreateFile(PChar(FileName), GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, 0, 0);
end;

//   F -    
function FileValid(F: TFile): Boolean;
begin
  Result := F <> NULL_FILE;
end;

//  
procedure FileClose(var F: TFile);
begin
  if FileValid(F) then
  begin
    CloseHandle(F);
    F := NULL_FILE;
  end;
end;

//     
procedure FileFlush(F: TFile);
begin
  FlushFileBuffers(F);
end;

//    
function FileWrite(F: TFile; const Buf; Count: DWORD): DWORD;
begin
  WriteFile(F, Buf, Count, Result, nil);
end;

//    
function FileRead(F: TFile; var Buf; Count: DWORD): DWORD;
begin
  ReadFile(F, Buf, Count, Result, nil);
end;

procedure FileSeek(F: TFile; Pos: Integer);
begin
  SetFilePointer(F, Pos, nil, FILE_BEGIN);
end;

function FileSize(F: TFile): DWORD;
begin
  Result := GetFileSize(F, nil);
end;

function FilePos(F: TFile): DWORD;
begin
  Result := SetFilePointer(F, 0, nil, FILE_CURRENT);
end;

function FileExists(FileName: PChar): Boolean;
var
  F : TFile;
begin
  F := FileOpen(FileName, False);
  Result := FileValid(F);
  if Result then
    FileClose(F);
end;

// TInterface
constructor TInterface.CreateEx;
begin
  inherited Create;
end;

// TIntStatic
function TIntStatic.QueryInterface(const IID: TGUID; out Obj): HResult;
begin
  Result := 0;
end;

function TIntStatic._AddRef: Integer;
begin
  Result := 0;
end;

function TIntStatic._Release: Integer;
begin
  Result := 0;
end;

// TFileStream
constructor TFileStream.Create(FileName: string);
begin
  F := FileOpen(PChar(FileName));
end;

destructor TFileStream.Destroy;
begin
  if Valid then
    FileClose(F);
end;

function TFileStream.Valid: Boolean;
begin
  Result := FileValid(F);
end;

procedure TFileStream.Seek(Pos: Integer);
begin
  FileSeek(F, Pos);
end;

function TFileStream.Read(var Buf; Count: Integer): Integer;
begin
  try
    Result := FileRead(F, Buf, Count);
  except
    Result := 0;
  end;
end;

function TFileStream.Size: Integer;
begin
  Result := FileSize(F);
end;

function TFileStream.Pos: Integer;
begin
  Result := FilePos(F);
end;

// TMemoryStream
constructor TMemoryStream.Create(Mem: Pointer; Size: Integer);
begin
  FMem  := Mem;
  FSize := Size;
end;

function TMemoryStream.Valid: Boolean;
begin
  Result := FMem <> nil;
end;

procedure TMemoryStream.Seek(Pos: Integer);
begin
  FPos := Pos;
end;

function TMemoryStream.Read(var Buf; Count: Integer): Integer;
begin
  if FSize - FPos < Count then
    Result := FSize - FPos
  else
    Result := Count;

  try
    Move(Pointer(Integer(FMem) + FPos)^, Buf, Result);
    FPos := FPos + Result;
  except
    Result := 0;
  end;
end;

function TMemoryStream.Size: Integer;
begin
  Result := FSize;
end;

function TMemoryStream.Pos: Integer;
begin
  Result := FPos;
end;

///////////////////////////////////////////////////////////////

function StrLen(const Str: PChar): Cardinal;
asm
  {Check the first byte}
  cmp byte ptr [eax], 0
  je @ZeroLength
  {Get the negative of the string start in edx}
  mov edx, eax
  neg edx
  {Word align}
  add eax, 1
  and eax, -2
@ScanLoop:
  mov cx, [eax]
  add eax, 2
  test cl, ch
  jnz @ScanLoop
  test cl, cl
  jz @ReturnLess2
  test ch, ch
  jnz @ScanLoop
  lea eax, [eax + edx - 1]
  ret
@ReturnLess2:
  lea eax, [eax + edx - 2]
  ret
@ZeroLength:
  xor eax, eax
end;

function StrLComp(const Str1, Str2: PChar; MaxLen: Cardinal): Integer; assembler;
asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        MOV     EDI,EDX
        MOV     ESI,EAX
        MOV     EBX,ECX
        XOR     EAX,EAX
        OR      ECX,ECX
        JE      @@1
        REPNE   SCASB
        SUB     EBX,ECX
        MOV     ECX,EBX
        MOV     EDI,EDX
        XOR     EDX,EDX
        REPE    CMPSB
        MOV     AL,[ESI-1]
        MOV     DL,[EDI-1]
        SUB     EAX,EDX
@@1:    POP     EBX
        POP     ESI
        POP     EDI
end;

function StrLCopy(Dest: PChar; const Source: PChar; MaxLen: Cardinal): PChar; assembler;
asm
        PUSH    EDI
        PUSH    ESI
        PUSH    EBX
        MOV     ESI,EAX
        MOV     EDI,EDX
        MOV     EBX,ECX
        XOR     AL,AL
        TEST    ECX,ECX
        JZ      @@1
        REPNE   SCASB
        JNE     @@1
        INC     ECX
@@1:    SUB     EBX,ECX
        MOV     EDI,ESI
        MOV     ESI,EDX
        MOV     EDX,EDI
        MOV     ECX,EBX
        SHR     ECX,2
        REP     MOVSD
        MOV     ECX,EBX
        AND     ECX,3
        REP     MOVSB
        STOSB
        MOV     EAX,EDX
        POP     EBX
        POP     ESI
        POP     EDI
end;

procedure StrDispose(Str: PChar);
begin
  if Str <> nil then
  begin
    Dec(Str, SizeOf(Cardinal));
    FreeMem(Str, Cardinal(Pointer(Str)^));
  end;
end;

function StrAlloc(Size: Cardinal): PChar;
begin
  Inc(Size, SizeOf(Cardinal));
  GetMem(Result, Size);
  Cardinal(Pointer(Result)^) := Size;
  Inc(Result, SizeOf(Cardinal));
end;

function StrMove(Dest: PChar; const Source: PChar; Count: Cardinal): PChar;
begin
  Result := Dest;
  Move(Source^, Dest^, Count);
end;

function StrNew(const Str: PChar): PChar;
var
  Size: Cardinal;
begin
  if Str = nil then Result := nil else
  begin
    Size := StrLen(Str) + 1;
    Result := StrMove(StrAlloc(Size), Str, Size);
  end;
end;
///////////////////////////////////////////////////////////////

//===  /   ===-----------------------------
function GetExceptionObject(P: PExceptionRecord): TObject;
begin
  Result := TObject.Create;
end;

procedure ErrorHandler(ErrorCode: Byte; ErrorAddr: Pointer); export;
begin
  raise TObject.Create at ErrorAddr;
end;

procedure ExceptHandler(ExceptObject: TObject; ExceptAddr: Pointer); far;
begin
  //
end;


initialization
  ErrorProc      := ErrorHandler;
  ExceptProc     := @ExceptHandler;
  ExceptionClass := TObject;
  ExceptObjProc  := @GetExceptionObject;
end.
